﻿using Inet.Viewer.Data;
using Inet.Viewer.Resources;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace Inet.Viewer.WinForms.Prompt
{
    /// <summary>
    /// Base class for user controls which represents a prompt field.
    /// </summary>
    public class PromptControl : UserControl
    {
        /// <summary>
        /// The error provider to show errors during validation.
        /// </summary>
        protected ErrorProvider ErrorProvider { get; set; }

        /// <summary>
        /// The prompt data instance.
        /// </summary>
        protected PromptData PromptData { get; set; }

        /// <summary>
        /// The default control to show validation errors.
        /// </summary>
        protected Control ControlForErrors { get; set; }

        /// <summary>
        /// Gets or sets a flag indicating that the error provider should show errors.
        /// </summary>
        public bool ShowError { get; set; }

        /// <summary>
        /// Occurs when the input value was changed.
        /// </summary>
        internal event EventHandler ValueChanged;


        /// <summary>
        /// Gets or sets the current value of this prompt field.
        /// </summary>
        internal virtual PromptValue Value { 
            get {
                return null;
            }
            set
            {
            }
        }

        /// <summary>
        /// Default validation of the current prompt value. When an error occurs, it will be 
        /// displayed with the provider specified in property ErrorProvider at the control specified in 
        /// property ControlForErrors. These properties must be set before using the default
        /// validation.
        /// </summary>
        /// <returns>true if validation succeeded</returns>
        internal virtual bool ValidatePrompt()
        {
            if (ControlForErrors == null || ErrorProvider == null || PromptData == null)
            {
                return false;
            }
            ErrorProvider.Clear();
            try
            {
                return ValidatePrompt(Value, ControlForErrors);
            }
            catch (Exception exc)
            {
                if (ShowError)
                {
                    ErrorProvider.SetError(ControlForErrors, exc.Message);
                }
                ErrorMessage = exc.Message;
                return false;
            }
        }

        /// <summary>
        /// Validates the specified value. If the value is invalid, an error will be shown
        /// at the specified control.
        /// </summary>
        /// <param name="value">the value to validate</param>
        /// <param name="control">the control to show any errors</param>
        /// <returns></returns>
        protected bool ValidatePrompt(PromptValue value, Control control)
        {
            try
            {
                ErrorProvider.BlinkStyle = ErrorBlinkStyle.NeverBlink;
                if (!PromptData.WithinLimits(value))
                {
                    if (ShowError)
                    {
                        this.ErrorProvider.SetError(control, strings.Prompt_Error_OutOfRange + PromptData.RangeExplanationMsg);
                    }
                    ErrorMessage = strings.Prompt_Error_OutOfRange + PromptData.RangeExplanationMsg;
                    return false;
                }
                string err;
                if (value is RangePromptValue && (err = ((RangePromptValue)value).CheckRange()) != null)
                {
                    if (ShowError)
                    {
                        this.ErrorProvider.SetError(control, err);
                    }
                    ErrorMessage = err;
                    return false;
                }
                return true;
            }
            catch (Exception exc)
            {
                if (ShowError)
                {
                    this.ErrorProvider.SetError(control, exc.Message);
                }
                ErrorMessage = exc.Message;
                return false;
            }
        }

        /// <summary>
        /// A default handle changed implementation for change events from inner controlls. Validates the
        /// input and fires a value change event of this instance.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        protected void HandleChanged(object sender, EventArgs e)
        {
            ValidatePrompt();
            OnValueChanged();
        }

        /// <summary>
        /// Fires a value changed event.
        /// </summary>
        internal protected void OnValueChanged()
        {
            ShowError = true;
            if (ValueChanged != null)
            {
                ValueChanged.Invoke(this, new EventArgs());
            }
        }

        /// <summary>
        /// Updates the width of the specified combobox according to the widths of its
        /// entries.
        /// </summary>
        /// <param name="cmb">the combobox to auto-size</param>
        protected static void AutoSizeComboBox(ComboBox cmb)
        {
            int width = 0;
            Font font = cmb.Font;
            foreach (object o in cmb.Items)
            {
                string text = cmb.GetItemText(o);
                int w = TextRenderer.MeasureText(text, font).Width;
                if (w > width)
                {
                    width = w;
                }
            }
            cmb.Size = new Size(width + 40, cmb.Size.Height);
        }

        /// <summary>
        /// Gets or sets the last error message.
        /// </summary>
        public virtual string ErrorMessage { get; set; }
    }
}
